package ga.core.algorithm.interactive;

import ga.core.evaluation.IInteractiveFitnessEvaluator;
import ga.core.goperators.ICrossoverOp;
import ga.core.goperators.IMutationOp;
import ga.core.individual.IIndividual;
import ga.core.individual.IndividualList;
import ga.core.individual.population.IPopulation;
import ga.core.logging.IGALogger;
import ga.core.selection.ISelector;
import ga.core.validation.GAContext;
import ga.core.validation.IValidator;

import java.util.logging.Logger;

/**
 * Abstract class for an simple interactive genetic algorithm.
 * 
 * @param <T>
 *          The generic type of individuals.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public abstract class AbstractSIGA<T extends IIndividual<T>> implements
    ISIGA<T> {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(AbstractSIGA.class
      .getName());

  // for debugging conly
  private final Thread accessThread;

  private final IGALogger<T> gaLogger;

  private final IPopulation<T> population;
  private final ISelector<T> selector;
  private final IInteractiveFitnessEvaluator<T> evaluator;
  private final IMutationOp<T> mutationOp;
  private final ICrossoverOp<T> crossoverOp;
  private IValidator<T> validator;
  private final GAContext context = new GAContext();

  private final IndividualList<T> evaluatingIndividuals = new IndividualList<T>();

  private boolean validate;
  private final boolean useEliteStrategy;
  private int generation;

  /**
   * Initializes the abstract class.
   * 
   * @param population
   *          Population for the GA.
   * @param evaluator
   *          The automatic evaluator.
   * @param selector
   *          The selector.
   * @param mutateOperator
   *          The mutation operator.
   * @param crossoverOperator
   *          The crossover operator.
   * @param validator
   *          The validator
   * @param useEliteStrategy
   *          Use elite strategy or not.
   * @param gaLogger
   *          The ga logger.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public AbstractSIGA(final IPopulation<T> population,
      final IInteractiveFitnessEvaluator<T> evaluator,
      final ISelector<T> selector, final IMutationOp<T> mutateOperator,
      final ICrossoverOp<T> crossoverOperator, final IValidator<T> validator,
      final boolean useEliteStrategy, final IGALogger<T> gaLogger) {
    this.population = population;
    this.selector = selector;
    this.evaluator = evaluator;
    this.mutationOp = mutateOperator;
    this.crossoverOp = crossoverOperator;
    this.gaLogger = gaLogger;

    setValidator(validator);
    this.useEliteStrategy = useEliteStrategy;

    evaluator.addEvaluationListener(this);
    evaluator.setAlgorithm(this);

    population.setEvaluator(evaluator);

    this.accessThread = Thread.currentThread();
  }

  @Override
  public void init() {
    checkThread();

    population.initRandomly(validate ? validator : null, context);

    if (gaLogger != null) {
      gaLogger.evaluationStarted();
      gaLogger.populationInitiated(generation, population);
    }
  }

  /**
   * Helper method to check if the current thread is the creation thread.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  protected void checkThread() {
    if (!Thread.currentThread().equals(accessThread)) {
      LOGGER
          .warning("Multiple Threads are accessing this object! Creation thread: "
              + accessThread.getName()
              + " Current thread: "
              + Thread.currentThread().getName());
    }
  }

  /**
   * Getter for the ga logger. May only be called by subclasses.
   * 
   * @return The ga logger.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  protected IGALogger<T> getGALogger() {
    return gaLogger;
  }

  /**
   * Getter for the selector. May only be called by subclasses.
   * 
   * @return The selector.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  protected ISelector<T> getSelector() {
    return selector;
  }

  /**
   * Getter for the elite behavior. May only be called by subclasses.
   * 
   * @return <code>true</code> if should use elite strategy.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  protected boolean isUseEliteStrategy() {
    return useEliteStrategy;
  }

  @Override
  public ICrossoverOp<T> getCrossoverOp() {
    return crossoverOp;
  }

  @Override
  public IMutationOp<T> getMutationOp() {
    return mutationOp;
  }

  /**
   * Getter for the list of individuals, that are currently being evaluated. May
   * only be called by subclasses.
   * 
   * @return List of evaluating individuals.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  protected IndividualList<T> getEvaluatingIndividuals() {
    return evaluatingIndividuals;
  }

  /**
   * Getter for the validation behavior. May only be called by subclasses.
   * 
   * @return <code>true</code> if should validate individuals.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  protected boolean isValidate() {
    return validate;
  }

  /**
   * Getter for the validator. May only be called by subclasses.
   * 
   * @return The validator or <code>null</code> if not set.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  protected IValidator<T> getValidator() {
    return validator;
  }

  /**
   * Getter for the evaluator.
   * 
   * @return The interactive fitness evaluator.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  protected IInteractiveFitnessEvaluator<T> getEvaluator() {
    return evaluator;
  }

  @Override
  public void setValidate(final boolean validate) {
    if (validate && validator == null) {
      throw new RuntimeException("Error - no validator has been set");
    }

    this.validate = validate;
  }

  @Override
  public void setValidator(final IValidator<T> validator) {
    this.validator = validator;

    if (validator != null) {
      validate = true;
    }
  }

  @Override
  public IPopulation<T> getPopulation() {
    return population;
  }

  @Override
  public GAContext getContext() {
    return context;
  }

  @Override
  public int getGeneration() {
    return generation;
  }

  /**
   * Increments the current generation number. May only be called by subclasses.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  protected void incGeneration() {
    generation++;
  }

  @Override
  public void exit() {
    if (gaLogger != null) {
      gaLogger.exit();
    }
  }
}
